import Gtk from 'gi://Gtk?version=4.0';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import Gdk from 'gi://Gdk';

export const CrystalLauncher = GObject.registerClass(
    class CrystalLauncher extends Gtk.Window {
        _init(app) {
            super._init({ 
                application: app, 
                title: "Start",
                type: Gtk.WindowType.POPUP, // Geen randen, zwevend
                modal: true,
                resizable: false
            });

            this.add_css_class('crystal-launcher-window');
            this.set_default_size(640, 550);

            // Hoofdcontainer
            const mainBox = new Gtk.Box({ orientation: Gtk.Orientation.VERTICAL, spacing: 15 });
            mainBox.set_margin_top(20);
            mainBox.set_margin_bottom(20);
            mainBox.set_margin_start(20);
            mainBox.set_margin_end(20);

            // 1. Zoekbalk (Windows 11 stijl bovenin)
            this.searchEntry = new Gtk.SearchEntry({ placeholder_text: "Zoek naar apps, instellingen en documenten..." });
            this.searchEntry.connect('search-changed', () => this._filterApps());
            mainBox.append(this.searchEntry);

            // 2. Scrollbaar Grid voor Apps
            const scroll = new Gtk.ScrolledWindow();
            scroll.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC);
            scroll.set_vexpand(true);

            this.flowbox = new Gtk.FlowBox({
                valign: Gtk.Align.START,
                max_children_per_line: 6,
                min_children_per_line: 6,
                selection_mode: Gtk.SelectionMode.NONE,
                column_spacing: 10,
                row_spacing: 15
            });

            scroll.set_child(this.flowbox);
            mainBox.append(scroll);

            // 3. Footer (User profile & Power buttons)
            const footer = new Gtk.Box({ orientation: Gtk.Orientation.HORIZONTAL, spacing: 10 });
            footer.add_css_class('launcher-footer');
            
            const userLabel = new Gtk.Label({ label: GLib.get_user_name(), xalign: 0 });
            userLabel.add_css_class('user-label');
            footer.append(userLabel);

            // Spacer
            const spacer = new Gtk.Box({ hexpand: true });
            footer.append(spacer);

            // Power knop
            const powerBtn = new Gtk.Button({ icon_name: 'system-shutdown-symbolic' });
            powerBtn.add_css_class('power-button');
            powerBtn.connect('clicked', () => {
                // Roep standaard GNOME session dialog aan
                GLib.spawn_command_line_async('gnome-session-quit --power-off');
            });
            footer.append(powerBtn);

            mainBox.append(footer);
            this.set_child(mainBox);

            // Data laden
            this._loadApps();
            
            // Sluit menu als focus verliest (klik buiten menu)
            const controller = new Gtk.EventControllerFocus();
            controller.connect('leave', () => this.hide());
            this.add_controller(controller);
        }

        _loadApps() {
            this.allApps = Gio.AppInfo.get_all().filter(app => app.should_show());
            // Sorteer alfabetisch
            this.allApps.sort((a, b) => a.get_name().localeCompare(b.get_name()));

            this.allApps.forEach(app => {
                const item = this._createAppItem(app);
                this.flowbox.append(item);
            });
        }

        _createAppItem(app) {
            const btn = new Gtk.Button();
            btn.add_css_class('app-grid-button');
            
            const box = new Gtk.Box({ orientation: Gtk.Orientation.VERTICAL, spacing: 5 });
            
            // Icoon
            const icon = app.get_icon();
            const img = new Gtk.Image({ pixel_size: 48 });
            if (icon) img.set_from_gicon(icon);
            else img.set_from_icon_name("application-x-executable");
            
            // Naam
            const label = new Gtk.Label({ 
                label: app.get_name(), 
                max_width_chars: 10, 
                ellipsize: Pango.EllipsizeMode.END,
                wrap: false
            });
            label.add_css_class('app-label');

            box.append(img);
            box.append(label);
            btn.set_child(box);

            // Launch logic
            btn.connect('clicked', () => {
                const ctx = Gdk.Display.get_default().get_app_launch_context();
                app.launch([], ctx);
                this.hide();
            });

            // Bewaar referentie voor search
            btn._appName = app.get_name().toLowerCase();
            return btn;
        }

        _filterApps() {
            const query = this.searchEntry.get_text().toLowerCase();
            let child = this.flowbox.get_first_child();
            while (child) {
                const btn = child.get_child(); // De button in de flowbox child
                if (btn._appName.includes(query)) {
                    child.set_visible(true);
                } else {
                    child.set_visible(false);
                }
                child = child.get_next_sibling();
            }
        }

        toggle() {
            if (this.is_visible()) {
                this.hide();
            } else {
                // Centreer op scherm (of positioneer boven startknop in productie)
                this.present();
            }
        }
    }
);